proc New { } {
    global fileName typelist
    if [abandon] { return 1 }
    set out \
	[tk_getSaveFile -filetypes $typelist \
	     -title "New File" -initialfile $fileName]
    if {$out != ""} {
	set fileName $out
    } else {
	return 1
    }
    clearEntries
    setTitle $fileName
    return 0
}

proc clearEntries { } {
    global body input descriptions currentline 
    global oldvalue oldinsert oldline lines
    # delete everything in lists
    $body.values delete 0 end
    $body.variables delete 0 end
    catch { unset descriptions }
    catch { unset lines }
    catch { unset currentline }
    catch { unset oldline }
    catch { unset oldvalue }
    catch { unset oldinsert }
    # currentline might not be set
    EmptyInput $input
}

proc abandon { } {
    global hasChanged
    if {$hasChanged} {
	set choice [tk_messageBox -type yesno -default yes \
			-message "Abandon changes?" \
			-icon question]
	if { $choice == "no" } {
	    return 1
	}
    }
    set hasChanged 0
    return 0
}

proc traceChanges { name index op } {
    global menubar
    upvar $name var
    if {$name == "hasChanged" && $op == "w"} {
	    $menubar.mFile entryconfigure Save -state normal
	    $menubar.mFile entryconfigure "Save As*" -state normal
	if {$var} {
	} else {
	    $menubar.mFile entryconfigure Save -state disabled
	    $menubar.mFile entryconfigure "Save As*" -state disabled
	}
    }
}
proc Open { } {
    global fileName typelist
    set out \
	[tk_getOpenFile -filetypes $typelist \
	     -title "Open File" -initialfile $fileName]
    if {$out != ""} {
	set fileName $out
	return [reload]
    } else {
	return 1
    }
}

proc saveAs { } {
    global fileName
    set typelist {
	{"HTML Files" {".html" ".shtml"}}
	{"All Files" {"*"}}
    }
    set out \
	[tk_getSaveFile -filetypes $typelist \
	     -title "Save File As..." -initialfile $fileName]
    if {$out != ""} {
	set fileName $out
	save
    }
}

proc reload { } {
    global body
    if {[abandon]} { return 1 }
    clearEntries
    # load configuration
    if {[loadConfig $body.variables $body.values]} {
	return 1
    } else {
	focus $body.variables
	status_ok
	return 0
    }
}

proc save {} {
    global lines body input descriptions hasChanged fileName
    if {![info exists descriptions]} {
	return 1
    }
    if {![info exists fileName]} {
	saveAs
	return
    }
    ok $body $input
    if [catch {open $fileName w} configFile] {
	puts stderr "Cannot open $fileName"
	exit 1
    } else {
	set i 0
	# process all lines of file
	if {[info exists lines]} {
	    foreach { line } $lines {
		puts $configFile $line
	    }
	}
	foreach { description } $descriptions {
	    set out [ $body.variables get $i ]
	    append out "\t"
	    set value [$body.values get $i]
	    append out $value
	    if [string compare $description {}] {
		append out "\t\# "
		append out $description
	    }
	    regsub -all "\n" $out "\n\t" out
	    # multi line redef's for templates
	    puts $configFile $out
	    incr i
	}
	close $configFile
    }
    set hasChanged 0
    setTitle $fileName
}

proc quit { } {
    global hasChanged input body
    cancel $body $input

    if {$hasChanged} {
	set choice [tk_messageBox -type yesnocancel -default yes \
			-message "Save changes before quitting?" \
			-icon question]
	if { [string compare $choice "cancel"] } {
	    # choice != cancel
	    if { ![string compare $choice "yes"] } {
		save
	    }
	    Quit
	}
    } else {
	Quit
    }
}

proc handleSelection { line lbox input } {
    global descriptions currentline
    cancel $lbox $input
    if {$line == ""} { return -1 }
    SeeAndActivate $lbox $line
    $input.ne insert 0 [$lbox.variables get $line]
    PutInput2 [$lbox.values get $line]
    $input.se insert 0 [lindex $descriptions $line]
    set currentline $line
    status_ok
}

proc cancel { lbox input } {
    global currentline oldvalue insert descriptions
    if { [info exists currentline] } { 
	if { $insert } {
	    $lbox.variables delete $currentline
	    $lbox.values delete $currentline
	    set descriptions [lreplace $descriptions $currentline $currentline]
	    set descriptions [lreplace $descriptions 0 0]
	    # delete empty list entries
	    set insert 0
	}
	$lbox.variables selection set $currentline
	$lbox.values selection set $currentline
	$lbox.variables selection anchor $currentline
	$lbox.values selection anchor $currentline
	status_ok
    }
    EmptyInput $input
    focus $lbox.variables
}

proc undo { lbox input } {
    global oldline oldvalue oldinsert descriptions hasChanged
    if [info exists oldvalue] {
	set comm [lindex $oldvalue 2]
	if {![info exists oldinsert]} {
	    # undo change in entry
	    $lbox.variables delete $oldline
	    $lbox.values delete $oldline
	    set descriptions \
		[lreplace $descriptions $oldline $oldline $comm]
	    $lbox.variables insert $oldline [lindex $oldvalue 0]
	    $lbox.values insert $oldline [lindex $oldvalue 1]
	} else {
	    if {$oldinsert == 1} {
		# undo deletion: insert entry
		set descriptions \
		    [linsert $descriptions $oldline $comm]
		$lbox.variables insert $oldline [lindex $oldvalue 0]
		$lbox.values insert $oldline [lindex $oldvalue 1]
	    } else {
		# undo insertion: delete entry
		set descriptions \
		    [lreplace $descriptions $oldline $oldline]
		$lbox.variables delete $oldline
		$lbox.values delete $oldline
	    }
	}
	SeeAndActivate $lbox $oldline
	EmptyInput $input
	unset oldline
	unset oldvalue
	incr hasChanged -1
	catch {unset oldinsert}
    } else {
	status_warn "No undo information available."
    }
}

proc delete { lbox input } {
    global currentline oldline oldvalue oldinsert
    global insert descriptions hasChanged
    cancel $lbox $input
    if { [catch {set line $currentline}] } {
	set line [$lbox.variables curselection]
	# no entry selected, but cursor on entry
	if { $line == "" } {
	    return
	    # nothing selected at all, return
	}
    }
    # create empty entry in list below current entry; clear selection
    set oldvalue [list [$lbox.variables get $line] \
		      [$lbox.values get $line] [lindex $descriptions $line]]
    $lbox.variables delete $line
    $lbox.values delete $line
    set hasChanged 1
    set descriptions [lreplace $descriptions $line $line]
    set currentline $line
    set oldline $line
    set insert 0
    set oldinsert 1
    SeeAndActivate $lbox $line
    status_ok
}

proc focusInput1 { } {
    global input
    focus $input.ne
}

proc insert { lbox input } {
    global currentline insert descriptions oldinsert
    if { [catch {set line $currentline}] } {
	set line [$lbox.variables curselection]
	# no entry selected, but cursor on entry
	if { $line == "" } {
	    set line 0
	    # nothing selected at all, assume 0
	}
    }
    # create empty entry in list below current entry; clear selection
    cancel $lbox $input
    $lbox.variables insert $line ""
    $lbox.values insert $line ""
    set descriptions [linsert $descriptions $line "" ""]
    set currentline $line
    focus $input.ne
    set insert 1
    set oldinsert -1
    SeeAndActivate $lbox $line
    status_ok
}

proc ok { lbox input } {
    global descriptions oldline currentline oldvalue hasChanged insert
    set var [GetInput1]
    set val [GetInput2]
    set comm [GetInput3]
    if { $var == "" && $val == "" } { 
	cancel $lbox $input
	return 0 
    }
    if { [info exists currentline] } {
	set oldvar [$lbox.variables get $currentline]
	set oldval [$lbox.values get $currentline]
	set oldcomm [lindex $descriptions $currentline]
	# save old value from previously selected entry
	# = value before user input
	if { [string compare $oldvar $var] || \
		 [string compare $oldval $val] || \
		 [string compare $oldcomm $comm] } {
	    if [checkInput $lbox $var $val $currentline] {
		# check whether (possibly modified) variable name
		# matches another variable name from the list
		return 1
	    }
	    if {$insert} {
		set descriptions \
		    [lreplace $descriptions $currentline $currentline]
	    }
	    set oldvalue [list $oldvar $oldval $oldcomm]
	    set hasChanged 1
	    $lbox.variables delete $currentline
	    $lbox.variables insert $currentline $var
	    replaceValue $currentline $val
	    set descriptions \
		[lreplace $descriptions $currentline $currentline $comm]
	    status_ok
	}
	SeeAndActivate $lbox $currentline
	EmptyInput $input
	# empty input fields
	focus $lbox.variables
    } else {
	addEntry $lbox $input
    }
    if [info exists currentline] {
	set oldline $currentline
    }
    catch {unset currentline}
    set insert 0
}

proc addEntry { lbox input } {
    global currentline insert descriptions oldinsert
    if { [catch {set line $currentline}] } {
	set line [$lbox.variables curselection]
	# no entry selected, but cursor on entry
	if { $line == "" } {
	    set line 0
	    # nothing selected at all, assume 0
	}
    }
    set var [GetInput1]
    set val [GetInput2]
    if { $var == "" && $val == "" } { 
	cancel $lbox $input
	$lbox.variables delete $line
	$lbox.values delete $line
	if {[info exists descriptions]} {
	    set descriptions [lreplace $descriptions $line $line]
	}
	return 0
    }
    
    # check whether variable name already exists
    if {[checkInput $lbox $var $val -1]} { return 1 }
    if { !$insert } {
	$lbox.variables insert $line ""
	$lbox.values insert $line ""
	if {![info exists descriptions]} {
	    set descriptions ""
	} else {
	    set descriptions [linsert $descriptions $line "" ""]
	}
	set currentline $line
	set insert 1
	set oldinsert -1
    }
    ok $lbox $input
    SeeAndActivate $lbox $line
    status_ok
    # don't do default code (paste Xselection)
}

proc status_warn { msg } {
    status $msg
    status_col Red
}

proc status_ok {} {
    status "Select the setting you want to modify, or enter a new one."
    status_col Black
}

proc status { msg } {
    .status.msg config -text $msg
    # .status.msg must have been created beforehand
}

proc status_col { col } {
    .status.msg config -fg $col
    # .status.msg must have been created beforehand
}

proc initBindings { body input } {
    global tcl_platform
    # accelerators
    event add <<Reload>> <Meta-r> <Alt-r> <Control-r> <Command-r>
    event add <<Save>> <Meta-s> <Alt-s> <Control-s> <Command-s>
    event add <<Exit>> \
	<Meta-x> <Alt-x> <Control-x> <Control-c> <Command-period> <Command-x>
    event add <<Add>> <Meta-Insert> <Alt-Insert> <Control-Insert> <Command-Insert>
    event add <<Delete>> <Meta-Delete> <Alt-Delete> <Control-Delete> <Command-Delete>
    event add <<Cancel>> <Escape>
    event add <<Undo>> <Alt-z> <Control-z> <Command-z> <Control-BackSpace> <Control-slash>
    event add <<FocusList>> <Meta-l> <Alt-l> <Control-l> <Command-l>
    event add <<Select>> <space> <Return> <Select> <Control-slash> <F2>
    event add <<Open>> <Meta-o> <Alt-o> <Control-o> <Command-o>
    event add <<SaveAs>> <Meta-a> <Alt-a> <Control-a> <Command-a>
    event add <<New>> <Meta-n> <Alt-n> <Control-n> <Command-n>
    bind . <<Open>> Open
    bind . <<SaveAs>> saveAs
    bind . <<New>> New
    bind . <<FocusVar>> { focus $input.ne }
    bind . <<FocusVal>> { focusInput2 }
    bind . <<Reload>> reload
    bind . <<Save>> save
    bind . <<Exit>> quit
    bind . <<Cancel>> { cancel $body $input }
    bind . <<Undo>> { undo $body $input }
    bind . <<FocusList>> { focus $body.variables }
    set platform $tcl_platform(platform)
    if { ![string compare $platform "unix"] } {
	set accKey "Alt"
    } elseif { ![string compare $platform "macintosh"] } {
	set accKey "Command"
    } else {
	set accKey "Ctrl"; # for that other OS :(
    }
    append accKey "+"
    return $accKey
}

proc createBody { body input padding title1 title2 } {
    # create body frame content
    label $body.left -text $title1 -underline 0
    label $body.right -text $title2
    grid $body.left $body.right
    # create a pair of widgets and a scrollbar
    listbox $body.variables \
	-yscrollcommand [list Scroll_Set $body.scroll \
	[list grid $body.scroll -row 1 -column 2 -sticky ns]] \
	-exportselection false
    listbox $body.values -yscrollcommand [list Scroll_Set $body.scroll \
	[list grid $body.scroll -row 1 -column 2 -sticky ns]]
    scrollbar $body.scroll -orient vertical \
	-command [list BindYView [list $body.variables $body.values]]
    #    $body.values configure -setgrid true
    # gridding is still a bit buggy
    # combine selections of both lists
    foreach l [list $body.variables $body.values] {
	bind $l <Button-1> \
	[list Bind_Select %y $body $input]
	bind $l <<Select>> \
	[list KeySelect $l $body $input]
	bind $l <B1-Motion> \
	[list Bind_Select %y $body $input]
	bind $l <B2-Motion> \
	[list BindDragto %x %y $body.variables $body.values]
	bind $l <Button-2> \
	[list BindMark %x %y $body.variables $body.values]
	bind $l <Shift-B1-Motion> {}
	bind $l <Shift-Button-1> {}
	bind $l <Delete> {delete $body $input; break }
	bind $l <Insert> {insert $body $input; break }
	bind $l <Up> [list fixSelection $l $body $input -1]
	bind $l <Down> [list fixSelection $l $body $input 1]
    }
    grid $body.variables $body.values $body.scroll -sticky news -padx $padding
    grid rowconfigure $body 1 -weight 1
    grid columnconfigure $body 1 -weight 1
    Bind_Display $body
}
