#!/usr/bin/wish
###################################################
# Template editor, v1.0.1
# by Cyrille Artho, distributed under the GPL
# takes definitions like variable\tvalue[\tdescription]
# shows these entries graphically and lets the user change them

set dir "/usr/local/htmlplain"; # directory where the program is installed
set confdir "$env(HOME)/.htmlplain";     # directory with settings
set dotdir $confdir
set conffile "config.pl"
source [file join "$dir/twolists.tcl"]
source [file join "$dir/htmlinput.tcl"]
source [file join "$dir/common.tcl"]
# parts common to both variable and template editor

proc loadConfig { list1 list2 } {
    global descriptions lines hasChanged fileName insert
    set insert 0
    # insert = insertion mode
    set hasChanged 0
    if [catch {open [file join $fileName] r} configFile] {
	puts stderr "Cannot open $fileName"
	return 1
    } else {
	# read the file
	while {[gets $configFile line] >= 0} {
	    if { [regexp {^\#} $line] || \
		     [regexp {^//} $line]} {
		lappend lines $line
		continue
	    }
	    if { [regexp {^/\*} $line] } {
		set commentFlag 1
	    }
	    if {[info exists commentFlag]} {
		lappend lines $line
		if {[regexp {\*/$} $line]} {
		    # ending comment
		    unset commentFlag
		}
		continue
	    }
	    if {[string trim $line] == ""} { continue }
	    if [regexp "(\[^\t\]*)\t(.*)" $line match variable value] {
		if {[regexp "(\[^\t\]*)\t(.*)" \
			 $value match value comment]} {
		    regsub {^\# *} $comment {} comment
		    regsub {^// *} $comment {} comment
		} else {
		    set comment ""
		}
		if {![regexp {^&.*;$} $variable]} {
		    if {$variable == ""} {
			if {![regsub {\\$} $newval {} newval]} {
			    append newval "\n"
			}
			append newval $value
			set newcom $comment
		    } else {
			# new definition, store old one
			if {[info exists newval]} {
			    $list1 insert end [rmBrackets $newvar]
			    $list2 insert end [trimBrackets $newval]
			    lappend descriptions $newcom
			}
			# new entry
			set newval $value
			set newvar $variable
			set newcom $comment
		    }
		} else {
		    # &entities; only one line
		    $list1 insert end $variable
		    $list2 insert end $value
		    lappend descriptions $comment
		}
		# settings are now stored in lists variables, values, comments
	    }
	}
	# last entry
	if {[info exists newval]} {
	    $list1 insert end [rmBrackets $newvar]
	    $list2 insert end [trimBrackets $newval]
	    lappend descriptions $newcom
	}
	close $configFile
    }
    setTitle $fileName
    return 0
}

proc setTitle { filename } {
    if {$filename != ""} {
	wm title . "Template editor - $filename"
    } else {
	wm title . "Template editor"
    }
}

proc rmBrackets { text } {
    regsub {^<*(.*)>*$} $text {\1} text
    set text [string trim $text]
    return $text
}

proc trimBrackets { text } {
    regsub {^<*(.*)>*$} $text {\1} text
    set text [string trim $text]
    set out "<"
    append out $text
    append out ">"
    regsub {>>*$} $out {>} out
    return $out
}

proc replaceValue { currentline val } {
    global body
    set value [$body.variables get $currentline]
    $body.values delete $currentline
    if { ![regexp {^&} $value] } {
	set val [trimBrackets $val]
    }
    $body.values insert $currentline $val
}

proc GetInput1 { } {
    global input
    set var [$input.ne get]
    return $var
}

proc GetInput2 { } {
    global input
    set val [copyBuffer $input.me.t]
    return $val
}

proc GetInput3 { } {
    global input
    return [$input.se get]
}

proc PutInput2 { text } {
    global input
    set t $input.me.t
    $t insert 1.0 $text
    ColorAll $t
}

proc focusInput2 { } {
    global input
    focus $input.me.t
}

proc EmptyInput { input } {
    $input.ne delete 0 end
    cutBuffer $input.me.t
    $input.se delete 0 end
}

proc checkInput { lbox var val currentline } {
    global insert
    global input
    if {$var == "" && $val == ""} { return 1 }
    if {$var == "" } {
	focus $input.ne
	status_warn "Please enter an HTML tag or entity."
	# force user to complete all fields
	return 1
    }
    if {$val== "" } {
	focusInput2
	status_warn "Please enter a redefinition."
	# force user to complete all fields
	return 1
    }
    for {set i 0} {$i < [$lbox.variables index end]} {incr i} {
	set tmpvar [$lbox.variables get $i]
	if { ![string compare $tmpvar $var] } {
	    if {$i != $currentline || $insert} {
		status_warn "Tag or entity already exists, choose a different one."
		return 1
	    }
	}
    }
    return 0
}

proc Quit { } {
    global dotdir
    # save workspace state and really quit
    if [catch {open [file join "$dotdir/.template"] w} configFile] {
#	puts stderr "Cannot open $dotdir/.template"
	exit 1
    } else {
	puts $configFile [wm geometry . ]
	close $configFile
	exit
    }
}

proc about { } {
    tk_messageBox -type ok -default ok \
	-message "Template editor 1.0.1\nWritten by Cyrille Artho" \
	-icon info
}

proc help {} {
    global dir
    set help [file join "$dir/help/config/help.shtml"]
    if [catch {exec "netscape" $help &} ] {
	 tk_messageBox -type ok -default ok \
	-message "Could not start Netscape.\nPlease open the help page 'help.html' in Netscape." \
	-icon error
    }
}

proc createMainWindow { } {
    global body input fileName typelist dotdir menubar
    wm title . "Template Editor"
    # create frames to hold content
    frame .input
    frame .body
    frame .status
    set menubar .menubar
    set input .input
    set body .body
    set status .status
    set padding 2
    event add <<FocusVar>> <Meta-t> <Alt-t> <Control-t> <Command-t>
    event add <<FocusVal>> <Meta-d> <Alt-d> <Control-d> <Command-d>
    set accKey [initBindings $body $input]
    # create menu
    menu $menubar
    . config -menu $menubar
    foreach { name } \
    { File Edit Help } {
	set $name [menu $menubar.m$name]
	$menubar add cascade -label $name \
	    -underline 0 -menu $menubar.m$name
    }

    foreach { name label pos acc } \
	{ New "New..." 0 N Open "Open..." 0 O reload Revert 0 R \
	      save Save 0 S saveAs "Save As..." 5 A quit Exit 1 X } {
	$File add command -label $label \
	    -underline $pos -command $name -accelerator "$accKey$acc"
	}
    set accK $accKey
    append accK "Z"
    $Edit add command -label Undo -accelerator $accK \
	-underline 0 -command {undo $body $input }
    $Edit add separator
    foreach { name label pos acc } \
	{ {addEntry $body $input} "Add Entry" 0 Insert \
	      {delete $body $input} "Delete Entry" 0 Delete } {
	$Edit add command -label $label -accelerator "$accKey$acc" \
	    -underline $pos -command $name
	}
    foreach { name label pos acc } \
	{ about About... 0 "" help Help 0 F1} {
	    if {[string compare $acc ""]} {
		$Help add command -label $label -accelerator "$acc" \
		    -underline $pos -command $name
	    } else {
		$Help add command -label $label \
		    -underline $pos -command $name
	    }
	}

    # input frame
    label $input.nw -text "Tag <...>" -justify left -underline 0
    entry $input.ne -relief sunken
    set text $input.me
    frame $text
    TextInit $text 42 6
    TextStyles $text.t -quote blue -html "#8000b0" \
	-ent "#008000" -normal black -err red -comment orange
    label $input.mw -text Redefinition -justify left -underline 2
    label $input.sw -text Comment -justify left
    entry $input.se -relief sunken
    button $input.ok -text Ok -command [list ok $body $input] -underline 1
    set e e
    set w w
    foreach ypos { n m s } {
	# grid row
	grid $input.$ypos$w $input.$ypos$e -padx $padding -pady $padding -sticky w
	grid $input.$ypos$e -sticky ew

	# popup menu
	set popup .popup$ypos
	menu $popup
	$popup add command -label "Cancel changes" -command { cancel $body $input }
	bind $input.$ypos$e <Button-3> [list tk_popup $popup %X %Y]
    }

    grid columnconfigure $input 1 -weight 1
    grid $input.ok -row 1 -column 2 -sticky news -padx 2
    event add <<Ok>> <Control-k> <Alt-k> <Meta-k> \
	<Return> <Meta-Return> <Alt-Return> <Control-Return> <Command-Return>
    bind . <<Ok>> {ok $body $input; break }
    bind $input.ne <<Add>> {addEntry $body $input; break }
    bind $input.me <<Add>> {addEntry $body $input; break }
    bind $input.se <<Add>> {addEntry $body $input; break }
    bind $input.ne <<Delete>> {delete $body $input; break }
    bind $input.me <<Delete>> {delete $body $input; break }
    bind $input.se <<Delete>> {delete $body $input; break }

    createBody $body $input $padding "List of tags" Redefinition
    # create status frame content
    label $status.msg -justify left \
	-text "Please wait while loading the template ..."

    pack $status.msg -side left -fill x -anchor w -padx $padding  -pady $padding
    # pack frames
    #pack $menubar -fill x -padx $padding  -pady $padding
    pack $input -fill x -anchor w
    pack $body -expand true -fill both
    pack $status -fill both -anchor w
    
    set typelist {
	{"HTML Files" {".html" ".shtml"}}
	{"All Files" {"*"}}
    }
    wm protocol . WM_DELETE_WINDOW quit
    wm minsize . 25 10
    # load workspace state
    if  { ![catch {open [file join "$dotdir/.template"] r} configFile]} {
	wm geometry . [gets $configFile]
	close $configFile
    }
    # load configuration
    if {[info exists fileName]} {
	if {[loadConfig $body.variables $body.values]} {
	    if {[Open]} {
		clearEntries
	    }
	}
    } else {
	set fileName [file join "variables.txt"]
	clearEntries
    }
    status_ok
    setTitle $fileName
}

if {[file exists [file join "./$conffile"]]} {
    set confdir "."
    # prefer local config file
}
if [catch {open [file join "$confdir/$conffile"] r} configFile] {
	puts stderr "Cannot open $confdir/$conffile"
	return 1
    } else {
	while {[gets $configFile line] >= 0} {
	    if {[regexp "templateFile *= *'(.*)'" \
		     $line -match fileName]} {
	    }
	}
    }
createMainWindow
focus $body.variables
trace variable hasChanged w traceChanges
