#!/usr/bin/wish

proc Scroll_Set {scrollbar geoCmd offset size} {
    if {$offset != 0.0 || $size != 1.0} {
	eval $geoCmd			;# Make sure it is visible
	$scrollbar set $offset $size
    } else {
	set manager [lindex $geoCmd 0]
	$manager forget $scrollbar
	# hide it
    }
}

proc copyBuffer { t } {
    foreach {key value index} [$t dump -text 1.0 "end"] {
	append text $value
    }
    if {![info exists text]} { set text "" }
    regsub "\n$" $text {} text
    return $text
}

proc cutBuffer { t } {
    clear $t 1.0 [$t index "end"]
    $t tag remove html 1.0 [$t index "end"]
    foreach {key value index} [$t dump -text 1.0 "end"] {
	append text $value
    }
    $t delete 1.0 "end"
    if {![info exists text]} { set text "" }
    return $text
}

proc clear { t start end } {
    # clears all tags but HTML tags from text
    $t tag remove quote $start $end
    $t tag remove ent $start $end
    $t tag remove err $start $end
    $t tag remove comment $start $end
}

proc getSelection { t } {
    set selection [$t tag ranges sel]
    if {[string compare $selection ""]} {
	# selected text to paste
	set sel_list [split $selection " "]
	set open [lindex $sel_list 0]
	set close [lindex $sel_list 1]
	foreach {key value index} [$t dump -text $open $close] {
	    append text $value
	}
    }
    if {[info exists text]} {
	return [list $text $open $close]
    } else {
	return [list {} {} {}]
    }
}

proc paste { t } {
    global checkRegexp
    set start [$t index "insert linestart"]
    set text [lindex [getSelection $t] 0]
    if {$text == ""} {
	catch {set text [selection get -selection CLIPBOARD]}
    }
    $t insert insert $text
    set end [$t index "insert lineend"]
    $t tag remove html $start $end
    clear $t $start $end
    syntaxColor $t $start $end $checkRegexp
    # do not carry out default action (insert again)
}

proc cut { t } {
    global checkRegexp
    set selection [getSelection $t]
    set text [lindex $selection 0]
    set open [lindex $selection 1]
    set close [lindex $selection 2]
    if {$text != ""} {
	clipboard clear
	clipboard append $text
	$t delete $open $close
	set start [$t index "insert linestart"]
	set end [$t index "insert lineend"]
	$t tag remove html $start $end
	clear $t $start $end
	syntaxColor $t $start $end $checkRegexp
    }
    # do not carry out default action (delete again)
}

proc check { t key } {
    global checkRegexp
    set start [$t index "insert linestart"]
    set end [$t index "insert lineend +1 chars"]
    if {[string compare $key ""]} {
	$t insert insert $key
    } else {
	# delete selection, if any
	cut $t
    }
    # insert key now, not later when default handler is carried out
    
    $t tag remove html $start $end
    clear $t $start $end
    syntaxColor $t $start $end $checkRegexp
    # do not carry out default action (insert character again)
}

proc syntaxColor { t start end regexp } {
    # start search
    set open [$t search -regexp $regexp $start $end];
    while {[string compare $open ""]} { 
	# symbol found
	set opening [lindex [$t dump -text "$open" "$open +1 chars"] 1]
	set closing [endSym $opening]
	set close [$t search -- $closing "$open +1 chars" $end];
	if {[string compare $close ""]} {
	    set kind [kind $opening]
	    $t tag remove html "$open +1 chars" "$close"
	    clear $t $open "$close +1 chars"
	    $t tag add $kind $open "$close +1 chars"
	    if {![string compare $kind "html"]} {
		# check for HTML comment
		set text \
		    [lindex [$t dump -text "$open +1 chars" "$open +2 chars"] 1]
		if {$text == "!"} {
		    $t tag remove html $open "$close +1 chars"
		    $t tag add comment $open "$close +1 chars"
		} else {
		    # allow stuff in quotes within HTML tags
		    set start [$t index "$open +1 chars"]
		    syntaxColor $t $start $close "\""
		    errorColor $t $start $close "&"
		}
	    } else {
		if {![string compare $kind "quote"]} {
		    set newregexp "\[<&\]"
		} else {
		    set newregexp "\[<\"\]"
		}
		errorColor $t $start $close $newregexp
	    }
	    set start [$t index "$close +1 chars"]
	    # search for next syntactical entity
	} else {
	    set start [$t index "$open +1 chars"]
	}
	set open [$t search -regexp $regexp $start $end];
    }
}

proc errorColor { t start end regexp } {
    # almost like syntaxColor, but not recursive
    # start search
    set open [$t search -regexp $regexp $start $end];
    while {[string compare $open ""]} { 
	# symbol found
	set opening [lindex [$t dump -text "$open" "$open +1 chars"] 1]
	set closing [endSym $opening]
	set close [$t search -- $closing "$open +1 chars" $end];
	if {[string compare $close ""]} {
	    $t tag remove html "$open +1 chars" $close
	    clear $t "$open +1 chars" "$close"
	    $t tag add err $open "$close +1 chars"
	    set start [$t index "$close +1 chars"]
	    # search for next syntactical entity
	} else {
	    set start [$t index "$open +1 chars"]
	}
	set open [$t search -regexp $regexp $start $end];
    }
}

proc endSym { char } {
    # return closing character/tag for given character
    if {![string compare $char "<"]} {
	return ">"
    } elseif {![string compare $char "\""]} {
	return "\""
    } elseif {![string compare $char "&"]} {
	return ";"
    } else {
	# error
	return -1
    }
}

proc kind { char } {
    # return kind of syntax symbol (returns tag name used for syntax coloring)
    switch $char { 
	"<" { return "html" }
	">" { return "html" }
	"\"" { return "quote" }
	"&" { return "ent" } 
	";" { return "ent" }
	"!" { return "comment" }
	default {
	    # error
	    return -1
	}
    }
}

proc Return { t } {
    global checkRegexp
    set start [$t index "insert linestart"]
    set end [$t index "insert lineend +1 chars"]
    $t insert insert "\n"
    set end [$t index "$end lineend"]
    $t tag remove html $start $end
    clear $t $start $end
    syntaxColor $t $start [$t index "$start lineend"] $checkRegexp
    syntaxColor $t [$t index "$end linestart"] $end $checkRegexp
    $t see $end
}

proc del { t key } {
    set char [lindex [$t dump -text insert "insert +1 chars"] 1]
    $t delete "insert"
    set kind [kind $char]
    if {$kind != -1 || $char == "\n"} {
	check $t ""
    }
}

proc back { t key } {
    set char [lindex [$t dump -text "insert -1 chars" insert] 1]
    $t delete "insert -1 chars"
    set kind [kind $char]
    if {$kind != -1 || $char == "\n"} {
	check $t ""
    }
}

proc ColorAll { t } {
    global checkRegexp
    set end [$t index end]
    set current 1.0
    while {[$t compare $current < $end]} {
	syntaxColor $t $current $end $checkRegexp
	set current [$t index "$current +1 lines"]
    }
}

proc TextStyles { t args } {
    #defaults
    $t tag configure quote -foreground blue
    $t tag configure html -foreground "#8000b0"
    $t tag configure ent -foreground "#008000"
    $t tag configure err -foreground red
    $t tag configure comment -foreground orange
    $t tag configure normal -foreground black
    while {[regexp -- {^-([^ ]+) ([^ ]+) ?} $args -match option value]} {
	$t tag configure $option -foreground $value
	regsub -- {^-[^ ]+ [^ ]+ ?} $args {} args
    }
}

proc TextInit { f w h } {
    global t checkRegexp
    set checkRegexp "\[<&\"\]"
    # bindings execute in global context, therefore global t needed
    # create text widget and scrollbar
    text $f.t -wrap word -width $w -height $h \
	-yscrollcommand [list Scroll_Set $f.sy\
			     [list pack $f.sy -side right -fill y]]
    scrollbar $f.sy -orient vert -command [list $f.t yview]
    set t $f.t
    pack $f.sy -side right -fill y
    pack $f.t -side left -fill both -expand true

    # key bindings
    event add <<Cut>> <Control-x> <Control-Delete>  <Alt-Delete> \
	<F20> <Shift-Delete> <F2>
    event add <<Copy>> <Control-Insert>
    event add <<Paste>> <Control-v> <F18> <Shift-Insert> <F4> \
	<ButtonRelease-2> <Insert>
    event add <<Focus>> <Shift-Tab> <Control-Tab> <Meta-Tab> <Command-Tab>
    # bindings for any platform, UNIX, another OS, Mac, and UNIX mouse
    bind $t <<Paste>> { paste $t ; break }
    bind $t <<Copy>> { continue }
    # needs to be used to prevent behavior of Insert without modifier key
    bind $t <<Cut>> { cut $t ; break }
    bind $t <Return> { Return $t ; break }
    bind $t <quotedbl> {check $t %A; break }
    bind $t <Delete> { del $t %A; break }
    bind $t <BackSpace> { back $t %A; break }
    bind $t <greater> { check $t %A; break }
    bind $t <less> { check $t %A; break }
    bind $t <ampersand> { check $t %A; break }
    bind $t <semicolon> { check $t %A; break }
    bind $t <minus> { check $t %A; break }
    bind $t <exclam> { check $t %A; break }
    bind $t <<Focus>> { tk_focusNext %W }
    bind $t <Tab> { 
	puts stderr "Use Ctrl+Tab for focus traversal."
	tk_focusNext %W
	break 
    }
    # tab character is not allowed
}

proc example { } {
    # how to use - example:
    set text .text
    frame $text
    pack $text -side top -fill both -expand true
    TextInit $text 42 14
    TextStyles $text.t -quote blue -html "#8000b0" -ent "#008000" -normal black \
	-err red -comment orange
}
