#!/usr/bin/wish
###################################################
# Configuration tool for Perl files, v1.0.1
# by Cyrille Artho, distributed under the GPL
# takes either comments lines or assignements like
# $variable = 'value'; # description
# shows these entries graphically and lets the user change them

set dir "/usr/local/htmlplain"; # directory where the program is installed
set confdir "$env(HOME)/.htmlplain";     # directory with settings
set dotdir $confdir
set conffile "config.pl"
source [file join "$dir/twolists0.tcl"]
source [file join "$dir/tkgetdir.tcl"]

proc reload { } {
    global body input descriptions oldline lines
    # delete everything in lists
    $body.values delete 0 [llength $descriptions ]
    $body.variables delete 0 [llength $descriptions ]
    unset descriptions
    unset lines
    catch { unset oldline }
    # oldline might not be set
    EmptyInput $input
    # load configuration
    loadConfig $body.variables $body.values
    focus $body.variables
}

proc loadConfig { list1 list2 } {
    global descriptions lines hasChanged fileName
    global conffile confdir
    set fileName [file join "$confdir/$conffile"]
    set hasChanged 0
    if [catch {open $fileName r} configFile] {
	puts stderr "Cannot open $fileName"
	exit 1
    } else {
	# read the file
	while {[gets $configFile line] >= 0} {
	    lappend lines $line
	    if [regexp {(.*;.*)\#(.*)} $line match text comm] {
		regsub {^ *} $comm {} comment
		regexp {[$]([^=]+) *= *(.*);} $text match variable value
		regsub { *$} $variable {} variable
		regsub {^'} $value {} value
		regsub {'$} $value {} value
		lappend descriptions $comment
		$list1 insert end $variable
		$list2 insert end $value
		# settings are now stored in lists variables, values, comments
	    }
	}
	close $configFile
    }
}

proc save {} {
    global lines body input descriptions hasChanged fileName
    ok $body $input
    if [catch {open $fileName w} configFile] {
	puts stderr "Cannot open $fileName"
	exit 1
    } else {
	set i 0
	# process all lines of file
	foreach { line } $lines {
	    set out "$"
	    append out [ $body.variables get $i ]
	    # process comment-only lines
	    if { ![regexp {(\$[^ =]+)} $line] } {
		puts $configFile $line
	    } else {
		# matching entry
		append out " = '"
		append out [$body.values get $i]
		append out "'; "
		set out [ format "%-39s" $out ]
		# fill up with spaces
		append out "\# "
		append out [lindex $descriptions $i]
		puts $configFile $out
		incr i
	    }
	}
	close $configFile
    }
    set hasChanged 0
}

proc quit { } {
    global hasChanged input body
    cancel $body $input
    if {$hasChanged} {
	set choice [tk_messageBox -type yesnocancel -default yes \
			-message "Save changes before quitting?" \
			-icon question]
	if { [string compare $choice "cancel"] } {
	    # choice != cancel
	    if { ![string compare $choice "yes"] } {
		save
	    }
	    Quit
	}
    } else {
	Quit
    }
}

proc handleSelection { line lbox input status } {
    global descriptions oldline
    cancel $lbox $input
    SeeAndActivate $lbox $line
    # trying to activate the right line, does not work with last line...
    $input.ne config -state normal
    $input.ne insert 0 [$lbox.variables get $line]
    $input.ne config -state disabled
    $input.se config -state normal
    $input.se insert 0 [$lbox.values get $line]
    $status.msg config -text [lindex $descriptions $line]
    set oldline $line
    focus $input.se
}

proc EmptyInput { input } {
    $input.ne config -state normal
    $input.ne delete 0 end
    $input.ne config -state disabled
    $input.se config -state normal
    $input.se delete 0 end
    $input.se config -state disabled
}

proc cancel { lbox input } {
    global oldline
    if { [info exists oldline] } {
	SeeAndActivate $lbox $oldline
    }
    EmptyInput $input
    catch { unset oldline }
    focus $lbox.variables
}

proc ok { lbox input } {
    global oldline hasChanged
    # save old value from previously selected entry
    if { [info exists oldline] } {
	set value [$input.se get]
	set old [$lbox.values get $oldline]
	# value before user input
	if { [string compare $old $value] } {
	    set hasChanged 1
	    $lbox.values delete $oldline
       	    $lbox.values insert $oldline $value
	}
	SeeAndActivate $lbox $oldline
	EmptyInput $input
	focus $lbox.variables
    }
    catch { unset oldline }
}

proc getDirectory { } {
    global input
    set entry $input.se
    set f "~"
    if {[$entry select present]} {
	set f [$entry get]
	set start [$entry index sel.first]
	set end [$entry index sel.last]
	set f [string range $f $start $end]
	set f [string trim $f]
	if {![file exists $f]} {
	    set f "~"
	}
    }
    # try later to set it to value after cursor, if valid ([^ ]+)
    set out \
	[tk_getDirectory -title "Choose directory" -initialdir $f]
    if {$out != ""} {
	catch {$entry delete $start $end}
    }
    $entry insert insert $out
}

proc Quit { } {
    global dotdir
    # save workspace state and really quit
    if [catch {open [file join "$dotdir/.config"] w} configFile] {
#	puts stderr "Cannot open $dotdir/.config"
	exit 1
    } else {
	puts $configFile [wm geometry . ]
	close $configFile
	exit
    }
}

proc help {} {
    global dir
    set help [file join "$dir/help/config/help.shtml"]
    if [catch {exec "netscape" $help &} ] {
	 tk_messageBox -type ok -default ok \
	-message "Could not start Netscape.\nPlease open the help page 'help.html' in Netscape." \
	-icon error
    }
}

proc createMainWindow { } {
    global body input oldline dotdir
    wm title . "Configuration Editor"
    # create frames to hold content
    frame .menubar
    frame .input
    frame .body
    frame .status
    set menubar .menubar
    set input .input
    set body .body
    set status .status
    set padding 2
    event add <<Reload>> <Meta-r> <Alt-r> <Control-r> <Command-r>
    event add <<Save>> <Meta-s> <Alt-s> <Control-s> <Command-s>
    event add <<Exit>> \
	<Meta-x> <Alt-x> <Control-x> <Control-c> <Command-period>
    event add <<Cancel>> <Escape>
    bind . <<Reload>> reload
    bind . <<Save>> save
    bind . <<Exit>> quit
    bind . <F1> help
    # create all menu buttons
    foreach { name label pos } \
    { reload Revert 0 save Save 0 quit Exit 1 } {
	button $menubar.$name -text $label -command $name -underline $pos
	pack $menubar.$name -side left
    }
    button $menubar.help -text Help -command help -underline -0
    pack $menubar.help -side right
    # input frame
    button $input.ok -text Ok -command [list ok $body $input] -underline 0
    label $input.nw -text Setting -justify left
    entry $input.ne -relief sunken -state disabled
    label $input.sw -text Value -justify left
    entry $input.se -relief sunken -state disabled
    grid $input.nw $input.ne -padx $padding -pady $padding -sticky w
    grid $input.sw $input.se -padx $padding -pady $padding -sticky w
    grid $input.ne -sticky ew
    grid $input.se -sticky ew
    grid columnconfigure $input 1 -weight 1
    grid $input.ok -row 0 -column 2 -rowspan 2 -sticky news -padx 2
    event add <<Ok>> \
	<Return> <Meta-Return> <Alt-Return> <Control-Return> <Command-Return>
    bind $input.se <<Ok>> [list ok $body $input]
    bind . <<Cancel>> [list cancel $body $input]
    # create body frame content
    label $body.left -text Setting
    label $body.right -text Value
    grid $body.left $body.right
    # create a pair of widgets and a scrollbar
    listbox $body.variables \
	-yscrollcommand [list Scroll_Set $body.scroll \
	[list grid $body.scroll -row 1 -column 2 -sticky ns]] \
	-exportselection false
    listbox $body.values -yscrollcommand [list Scroll_Set $body.scroll \
	[list grid $body.scroll -row 1 -column 2 -sticky ns]]
    scrollbar $body.scroll -orient vertical \
	-command [list BindYView [list $body.variables $body.values]]
    # $body.values configure -setgrid true
    # gridding still a bit buggy
    # combine selections of both lists
    event add <<Select>> <space> <Return> <Select> <Control-slash> <F2>
    foreach l [list $body.variables $body.values] {
	bind $l <Button-1> \
	[list Bind_Select %y $body $input $status]
	bind $l <<Select>> \
	[list KeySelect $l $body $input $status]
	bind $l <B1-Motion> \
	[list Bind_Select %y $body $input $status]
	bind $l <B2-Motion> \
	[list BindDragto %x %y $body.variables $body.values]
	bind $l <Button-2> \
	[list BindMark %x %y $body.variables $body.values]
	bind $l <Shift-B1-Motion> {}
	bind $l <Shift-Button-1> {}
	bind $l <Up> [list fixSelection $l $body $input $status -1]
	bind $l <Down> [list fixSelection $l $body $input $status 1]
    }
    grid $body.variables $body.values $body.scroll -sticky news -padx $padding
    grid rowconfigure $body 1 -weight 1
    grid columnconfigure $body 1 -weight 1
    Bind_Display $body
    # create status frame content
    label $status.msg -justify left \
	-text "Please wait while loading the configuration..."

    pack $status.msg -side left -fill x -anchor w -padx $padding  -pady $padding
    # pack frames
    pack $menubar -fill x -padx $padding -pady $padding
    pack $input -fill x -anchor w
    pack $body -expand true -fill both
    pack $status -fill both -anchor w
    
    # load configuration
    loadConfig $body.variables $body.values
    $status.msg config -text "Please select the setting you want to modify."
    # popup menu
    set popup .popup
    menu $popup
    $popup add command -label "Insert Directory..." \
	-command { getDirectory }
    $popup add command -label "Cancel changes" -command { cancel $body $input }
    bind $input.se <Button-3> [list tk_popup $popup %X %Y]
}

if {[file exists [file join "./$conffile"]]} {
    set confdir "."
    # prefer local config file
}
createMainWindow
wm protocol . WM_DELETE_WINDOW quit
wm minsize . 25 10
# load workspace state
if  { ![catch {open [file join "$dotdir/.config"] r} configFile]} {
    wm geometry . [gets $configFile]
    close $configFile
}
