#!/usr/bin/wish
###################################################
# Configuration tool for Perl files
# spreadsheet version, v1.0.1
# by Cyrille Artho, distributed under the GPL
# takes either comments lines or assignements like
# $variable = 'value'; # description
# shows these entries graphically and lets the user change them

set dir "/usr/local/htmlplain"; # directory where the program is installed
set confdir "$env(HOME)/.htmlplain";     # directory with settings
set dotdir $confdir
set conffile "config.pl"
source [file join "$dir/twocols.tcl"]
source [file join "$dir/twolists.tcl"]
source [file join "$dir/tkgetdir.tcl"]

proc reload { } {
    global body variables values descriptions oldline lines
    set frame .body.canvas.f
    # delete everything in lists
    unset descriptions
    unset variables
    unset values
    unset lines
    # load configuration
    set max [loadConfig]
    for {set i 0} {$i < $max} {incr i} {
	$frame.entry$i delete 0 end
	$frame.entry$i insert 0 [lindex $values $i]
    }
}

proc loadConfig { } {
    global lines variables values descriptions hasChanged
    global conffile confdir fileName
    set fileName [file join "$confdir/$conffile"]
    if [catch {open $fileName r} configFile] {
	puts stderr "Cannot open $fileName"
	exit 1
    } else {
	# read the file
	while {[gets $configFile line] >= 0} {
	    lappend lines $line
	    if [regexp {(.*;.*)\#(.*)} $line match text comm] {
		set value ""
		regsub {^ *} $comm {} comment
		regexp {[$]([^=]+) *= *(.*);} $text match variable value
		regsub { *$} $variable {} variable
		regsub {^'} $value {} value
		regsub {'$} $value {} value
		lappend descriptions $comment
		lappend variables $variable
		lappend values $value
		# settings are now stored in lists variables, values, comments
	    }
	}
	close $configFile
    }
    set hasChanged 0
    return [llength $variables]
}

proc save {} {
    global lines body variables values descriptions
    global hasChanged fileName oldline
    set frame .body.canvas.f
    if [info exists oldline] {
	ok $oldline $frame
    }
    if [catch {open $fileName w} configFile] {
	puts stderr "Cannot open $fileName"
	exit 1
    } else {
	set i 0
	# process all lines of file
	foreach { line } $lines {
	    set out "$"
	    append out [lindex $variables $i]
	    # process comment-only lines
	    if { ![regexp {(\$[^ =]+)} $line] } {
		puts $configFile $line
	    } else {
		# matching entry
		append out " = '"
		append out [lindex $values $i]
		append out "'; "
		set out [ format "%-39s" $out ]
		# fill up with spaces
		append out "\# "
		append out [lindex $descriptions $i]
		puts $configFile $out
		incr i
	    }
	}
	close $configFile
    }
    set hasChanged 0
}

proc quit { } {
    global hasChanged body oldline
    set frame .body.canvas.f
    if [info exists oldline] {
	cancel $oldline $frame
    }
    if {$hasChanged} {
	set choice [tk_messageBox -type yesnocancel -default yes \
			-message "Save changes before quitting?" \
			-icon question]
	if { [string compare $choice "cancel"] } {
	    # choice != cancel
	    if { ![string compare $choice "yes"] } {
		save
	    }
	    Quit
	}
    } else {
	Quit
    }
}

proc select { i frame max } {
    global oldline backgroundColor tcl_platform descriptions status
    if [info exists oldline] {
	ok $oldline $frame
	$frame.label$oldline configure -bg $backgroundColor
    }
    set platform $tcl_platform(platform)
    if { ![string compare $platform "unix"] } {
	$frame.label$i configure -bg "#c3c3c3"
    } else {
	$frame.label$i configure -bg systemHighlightText
    }
    $status.msg config -text [lindex $descriptions $i]
    set oldline $i
    focus $frame.entry$i
}

proc cancel { i frame } {
    global values
    $frame.entry$i delete 0 end
    $frame.entry$i insert 0 [lindex $values $i]
}

proc ok { i frame } {
    global values hasChanged
    set old [lindex $values $i]
    set new [$frame.entry$i get]
    if {[string compare $old $new]} {
	set hasChanged 1
    }
    set values [lreplace $values $i $i $new]
}

proc Quit { } {
    global dotdir
    # save workspace state and really quit
    if [catch {open [file join "$dotdir/.config2"] w} configFile] {
#	puts stderr "Cannot open $dotdir/.config2"
	exit 1
    } else {
	puts $configFile [wm geometry . ]
	close $configFile
	exit
    }
}

proc help {} {
    global dir
    set help [file join "$dir/help/config/help.shtml"]
    if [catch {exec "netscape" $help &} ] {
	 tk_messageBox -type ok -default ok \
	-message "Could not start Netscape.\nPlease open the help page 'help.html' in Netscape." \
	-icon error
    }
}

proc getDirectory { entry} {
    set f "~"
    if {[$entry select present]} {
	set f [$entry get]
	set start [$entry index sel.first]
	set end [$entry index sel.last]
	set f [string range $f $start $end]
	set f [string trim $f]
	if {![file exists $f]} {
	    set f "~"
	}
    }
    # try later to set it to value after cursor, if valid ([^ ]+)
    set out \
	[tk_getDirectory -title "Choose directory" -initialdir $f]
    if {$out != ""} {
	catch {$entry delete $start $end}
    }
    $entry insert insert $out
}

proc traceVar { name index op } {
    upvar $name var
    puts stdout "$op $name: $var"
}

proc createMainWindow { } {
    global body backgroundColor variables values dotdir status
    wm title . "Configuration Editor"
    # create frames to hold content
    set menubar .menubar
    set body .body
    #trace variable body rwu traceVar
    # body is somehow changes to something different (tcl bug)
    # "workaround" by using a hard coded value for .body
    set status .status
    frame $menubar
    frame .body
    frame $status
    set padding 2
    event add <<Reload>> <Meta-r> <Alt-r> <Control-r> <Command-r>
    event add <<Save>> <Meta-s> <Alt-s> <Control-s> <Command-s>
    event add <<Exit>> \
	<Meta-x> <Alt-x> <Control-x> <Control-c> <Command-period>
    bind . <<Reload>> reload
    bind . <<Save>> save
    bind . <<Exit>> quit
    bind . <F1> help
    # create all menu buttons
    foreach { name label pos } \
    { reload Revert 0 save Save 0 quit Exit 1 } {
	button $menubar.$name -text $label -command $name -underline $pos
	pack $menubar.$name -side left
    }
    button $menubar.help -text Help -command help -underline -0
    pack $menubar -fill x -padx $padding  -pady $padding
    # create body frame content
    ScrollCanvas .body
    # pack frames
    pack .body -expand true -fill both
    pack $status -fill both -anchor w
    pack $menubar.help -side right
    set max [loadConfig]
    Scrolled_EntrySet .body.canvas [list Setting Value] $variables $values
    set body .body
    bind $menubar.help <Tab> \
	[list keySelect 0 .body.canvas.f 0 $max]
    # create status frame content
    label $status.msg -justify left \
	-text "Please wait while loading the configuration..."

    pack $status.msg -side bottom -fill x -anchor w -padx $padding  -pady $padding
    # load configuration
    $status.msg config -text "Please select the setting you want to modify."
    set backgroundColor [lindex [.body.canvas.f.label0 configure -bg] 4]
}

if {[file exists [file join "./$conffile"]]} {
    set confdir "."
    # prefer local config file
}
createMainWindow
wm protocol . WM_DELETE_WINDOW quit
wm resizable . 0 1
# load workspace state
if  { ![catch {open [file join "$dotdir/.config2"] r} configFile]} {
    wm geometry . [gets $configFile]
    close $configFile
}
